/*
 * Copyright (C) 2009 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* this implements a sensors hardware library for the Android emulator.
 * the following code should be built as a shared library that will be
 * placed into /system/lib/hw/sensors.goldfish.so
 *
 * it will be loaded by the code in hardware/libhardware/hardware.c
 * which is itself called from com_android_server_SensorService.cpp
 */


/* we connect with the emulator through the "sensors" qemud service
 */
#define  SENSORS_SERVICE_NAME "sensors"

#define LOG_TAG "bc10_sensors"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <math.h>

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <cutils/log.h>
#include <cutils/native_handle.h>
#include <cutils/sockets.h>
#include <hardware/sensors.h>
#include "ami602.h"

#if 0
#define  D(...)  LOGD(__VA_ARGS__)
#else
#define  D(...)  ((void)0)
#endif

#define  E(...)  LOGE(__VA_ARGS__)

#include <hardware/qemud.h>

#define MAX_NUM_SENSORS 3

#define SUPPORTED_SENSORS  ((1<<MAX_NUM_SENSORS)-1)

#define  ID_BASE           SENSORS_HANDLE_BASE
#define  ID_ACCELERATION   (ID_BASE+0)
#define  ID_MAGNETIC_FIELD (ID_BASE+1)
#define  ID_ORIENTATION    (ID_BASE+2)
#define  ID_TEMPERATURE    (ID_BASE+3)

#define  SENSORS_ACCELERATION   (1 << ID_ACCELERATION)
#define  SENSORS_MAGNETIC_FIELD  (1 << ID_MAGNETIC_FIELD)
#define  SENSORS_ORIENTATION     (1 << ID_ORIENTATION)
#define  SENSORS_TEMPERATURE     (1 << ID_TEMPERATURE)

#define  ID_CHECK(x)  ((unsigned)((x)-ID_BASE) < 4)

#define  SENSORS_LIST  \
    SENSOR_(ACCELERATION,"acceleration") \
    SENSOR_(MAGNETIC_FIELD,"magnetic-field") \
    SENSOR_(ORIENTATION,"orientation") \

static const struct {
    const char*  name;
    int          id; } _sensorIds[MAX_NUM_SENSORS] =
{
#define SENSOR_(x,y)  { y, ID_##x },
    SENSORS_LIST
#undef  SENSOR_
};

#define AMI602_DEV "/dev/ami602"

static float one_rad = 180 / M_PI;

static float calc_azimuth(float cx, float cy, int reverse_direction)
{
    float azimuth = 0.0f;

    if (cx > 0 && cy > 0) {  // 0 < azimuth < 90 
       azimuth = fabs((atanf(cy/cx) * one_rad));
    }
    if (cx < 0 && cy > 0) {  // 90 < azimuth < 180 
       azimuth = 180.0f - fabs((atanf(cy/cx) * one_rad));
    }
    if (cx < 0 && cy < 0) {  // 180 < azimuth < 180 
       azimuth = 180.0f + fabs((atanf(cy/cx) * one_rad));
    }
    if (cx > 0 && cy < 0) {  // 270 < azimuth < 360 
       azimuth = 360.0f - fabs((atanf(cy/cx) * one_rad));
    }
    if (cx == 0 && cy == 0) {
        azimuth = 0.0f;
    }
    if (cx == 0 && cy > 0) {
        azimuth = 90.0f;
    }
    if (cx < 0 && cy == 0) {
        azimuth = 180.0f;
    }
    if (cx == 0 && cy < 0) {
        azimuth = 270.0f;
    }
    if (cx > 0 && cy == 0) {
        azimuth = 360.0f;
    } 

    if (reverse_direction) {
        if (azimuth >= 0.0f && azimuth <= 180.0f) {
            azimuth += 180.0f;
        } else if (azimuth > 180.0f && azimuth <= 360.0f) {
            azimuth -= 180.0f;
        }
    }

    return azimuth;
}

static float calc_pitch(float ay, float az)
{
    float pitch = 0.0f;

    //   TODO: implement convert method.

    return pitch;
}

static float calc_roll(float ay)
{
    float roll = 0.0f;

    //   TODO: implement convert method.

    return roll;
}


static const char*
_sensorIdToName( int  id )
{
    int  nn;
    for (nn = 0; nn < MAX_NUM_SENSORS; nn++)
        if (id == _sensorIds[nn].id)
            return _sensorIds[nn].name;
    return "<UNKNOWN>";
}

static int
_sensorIdFromName( const char*  name )
{
    int  nn;

    if (name == NULL)
        return -1;

    for (nn = 0; nn < MAX_NUM_SENSORS; nn++)
        if (!strcmp(name, _sensorIds[nn].name))
            return _sensorIds[nn].id;

    return -1;
}

/** SENSORS CONTROL DEVICE
 **
 ** This one is used to send commands to the sensors drivers.
 ** We implement this by sending directly commands to the emulator
 ** through the QEMUD channel.
 **/

typedef struct SensorControl {
    struct sensors_control_device_t  device;
    int                              fd;
    uint32_t                         active_sensors;
} SensorControl;

/* this must return a file descriptor that will be used to read
 * the sensors data (it is passed to data__data_open() below
 */
static native_handle_t*
control__open_data_source(struct sensors_control_device_t *dev)
{
    SensorControl*  ctl = (void*)dev;
    native_handle_t* handle;

    if (ctl->fd < 0) {
        ctl->fd = open(AMI602_DEV, O_RDWR);

    }
    D("%s: fd=%d", __FUNCTION__, ctl->fd);
    handle = native_handle_create(1, 0);
    handle->data[0] = dup(ctl->fd);
    return handle;
}

static int
control__activate(struct sensors_control_device_t *dev,
                  int handle,
                  int enabled)
{
    SensorControl*  ctl = (void*)dev;
    uint32_t        mask, sensors, active, new_sensors, changed;
    char            command[128];
    int             ret;

    D("%s: handle=%s (%d) fd=%d enabled=%d", __FUNCTION__,
        _sensorIdToName(handle), handle, ctl->fd, enabled);

    if (!ID_CHECK(handle)) {
        E("%s: bad handle ID", __FUNCTION__);
        return -1;
    }

    mask    = (1<<handle);
    sensors = enabled ? mask : 0;

    active      = ctl->active_sensors;
    new_sensors = (active & ~mask) | (sensors & mask);
    changed     = active ^ new_sensors;

    if (!changed)
        return 0;

    snprintf(command, sizeof command, "set:%s:%d",
                _sensorIdToName(handle), enabled != 0);

    if (ctl->fd < 0) {
        ctl->fd = open(AMI602_DEV, O_RDWR);
    }

    ctl->active_sensors = new_sensors;

    return 0;
}

static int
control__set_delay(struct sensors_control_device_t *dev, int32_t ms)
{
    SensorControl*  ctl = (void*)dev;
    char            command[128];

    D("%s: dev=%p delay-ms=%d", __FUNCTION__, dev, ms);

    return 0;
}

/* this function is used to force-stop the blocking read() in
 * data__poll. In order to keep the implementation as simple
 * as possible here, we send a command to the emulator which
 * shall send back an appropriate data block to the system.
 */
static int
control__wake(struct sensors_control_device_t *dev)
{
    SensorControl*  ctl = (void*)dev;
    D("%s: dev=%p", __FUNCTION__, dev);
    return 0;
}


static int
control__close(struct hw_device_t *dev) 
{
    SensorControl*  ctl = (void*)dev;
    close(ctl->fd);
    free(ctl);
    return 0;
}

/** SENSORS DATA DEVICE
 **
 ** This one is used to read sensor data from the hardware.
 ** We implement this by simply reading the data from the
 ** emulator through the QEMUD channel.
 **/
typedef struct SensorData {
    struct sensors_data_device_t  device;
    sensors_data_t                sensors[MAX_NUM_SENSORS];
    int                           events_fd;
    int                           stat;
    struct ami602_position        pos;
    int                           val;	// test 
} SensorData;

/* return the current time in nanoseconds */
static int64_t
data__now_ns(void)
{
    struct timespec  ts;

    clock_gettime(CLOCK_MONOTONIC, &ts);

    return (int64_t)ts.tv_sec * 1000000000 + ts.tv_nsec;
}

static int
data__data_open(struct sensors_data_device_t *dev, native_handle_t* handle)
{
    SensorData*  data = (void*)dev;
    int i;
    D("%s: dev=%p fd=%d", __FUNCTION__, dev, handle->data[0]);
    memset(&data->sensors, 0, sizeof(data->sensors));

    for (i=0 ; i<MAX_NUM_SENSORS ; i++) {
        data->sensors[i].vector.status = SENSOR_STATUS_ACCURACY_HIGH;
    }

    data->events_fd = dup(handle->data[0]);
    D("%s: dev=%p fd=%d (was %d)", __FUNCTION__, dev, data->events_fd, handle->data[0]);
    native_handle_close(handle);
    native_handle_delete(handle);
    return 0;
}

static int
data__data_close(struct sensors_data_device_t *dev)
{
    SensorData*  data = (void*)dev;
    D("%s: dev=%p", __FUNCTION__, dev);
    if (data->events_fd > 0) {
        close(data->events_fd);
        data->events_fd = -1;
    }
    return 0;
}

static int
data__poll(struct sensors_data_device_t *dev, sensors_data_t* values)
{
    SensorData*  data = (void*)dev;
    int fd = data->events_fd;
	int ret = 0;

    if (data->stat <= 0) {
            usleep(50000);
	    ret = ioctl(fd, AMI602_IOCPOSITION, &data->pos);
	    if (ret < 0) {
	        E("%s: ret=%d", __FUNCTION__, ret);
	        return -1;
	    }
	    data->stat = MAX_NUM_SENSORS;
		//	ID_ACCELERATION
	    data->sensors[0].sensor = ID_ACCELERATION;

		// Original formula
		// data->sensors[0].acceleration.x = (((float)(data->pos.accel_x - 2048) * GRAVITY_EARTH * -1.0) / 800.0f );
		// data->sensors[0].acceleration.y = (((float)(data->pos.accel_y - 2048) * GRAVITY_EARTH ) / 800.0f );
		// data->sensors[0].acceleration.z = (((float)(data->pos.accel_z - 2048) * GRAVITY_EARTH * -1.0) / 800.0f );
		data->sensors[0].acceleration.x = 25.1f - (float)data->pos.accel_x * 0.01225f;
		data->sensors[0].acceleration.y = (float)data->pos.accel_y * 0.01225f - 25.1f;
		data->sensors[0].acceleration.z = 25.1f - (float)data->pos.accel_z * 0.01225f;

	    //	ID_MAGNETIC_FIELD
	    //	AMI602 value: 1gauss = 600, 1uT = 6
	    data->sensors[1].sensor = ID_MAGNETIC_FIELD;
	    data->sensors[1].magnetic.x = (((float)(data->pos.mag_x - 2048) / 6.0f));
	    data->sensors[1].magnetic.y = (((float)(data->pos.mag_y - 2048) / 6.0f));
	    data->sensors[1].magnetic.z = (((float)(data->pos.mag_z - 2048) / 6.0f));
	    //	ID_ORIENTATION
	    data->sensors[2].sensor = ID_ORIENTATION;
	    data->sensors[2].orientation.azimuth = atan2( (data->sensors[1].magnetic.y * -1),  data->sensors[1].magnetic.x) * one_rad + 180;
	    data->sensors[2].orientation.pitch   = atan2( (data->sensors[0].acceleration.y * -1),  data->sensors[0].acceleration.z ) * one_rad;
	    data->sensors[2].orientation.roll    = atan2( (data->sensors[0].acceleration.x * -1),  data->sensors[0].acceleration.z ) * one_rad;
	}
    if (data->stat > 0) {
		switch (data->stat) {
		case 1:
		    *values = data->sensors[0];
		    ret = ID_ACCELERATION;
			break;
		case 2:
		    *values = data->sensors[1];
		    ret = ID_MAGNETIC_FIELD;
			break;
		case 3:
		    *values = data->sensors[2];
		    ret = ID_ORIENTATION;
			break;
		}
	}
    values->time = data__now_ns;
	data->stat--;
    
    return ret;
}

static int
data__close(struct hw_device_t *dev) 
{
    SensorData* data = (SensorData*)dev;
    if (data) {
        if (data->events_fd > 0) {
            //LOGD("(device close) about to close fd=%d", data->events_fd);
            close(data->events_fd);
        }
        free(data);
    }
    return 0;
}


/** MODULE REGISTRATION SUPPORT
 **
 ** This is required so that hardware/libhardware/hardware.c
 ** will dlopen() this library appropriately.
 **/

/*
 * the following is the list of all supported sensors.
 * this table is used to build sSensorList declared below
 * according to which hardware sensors are reported as
 * available from the emulator (see get_sensors_list below)
 *
 * note: numerical values for maxRange/resolution/power were
 *       taken from the reference AK8976A implementation
 */
static const struct sensor_t sSensorListInit[] = {
        { .name       = "BC10 6-axis Accelerometer",
          .vendor     = "BeatCraft. inc",
          .version    = 1,
          .handle     = ID_ACCELERATION,
          .type       = SENSOR_TYPE_ACCELEROMETER,
          .maxRange   = 2048.0f,
          .resolution = 1.0f,
          .power      = 1.0f,
          .reserved   = {}
        },

        { .name       = "BC10 3-axis Magnetic field sensor",
          .vendor     = "BeatCraft. inc",
          .version    = 1,
          .handle     = ID_MAGNETIC_FIELD,
          .type       = SENSOR_TYPE_MAGNETIC_FIELD,
          .maxRange   = 2048.0f,
          .resolution = 1.0f,
          .power      = 1.0f,
          .reserved   = {}
        },

        { .name       = "BC10 Orientation sensor",
          .vendor     = "BeatCraft. inc",
          .version    = 1,
          .handle     = ID_ORIENTATION,
          .type       = SENSOR_TYPE_ORIENTATION,
          .maxRange   = 360.0f,
          .resolution = 1.0f,
          .power      = 1.0f,
          .reserved   = {}
        },
#if 0
        { .name       = "Goldfish Temperature sensor",
          .vendor     = "The Android Open Source Project",
          .version    = 1,
          .handle     = ID_TEMPERATURE,
          .type       = SENSOR_TYPE_TEMPERATURE,
          .maxRange   = 80.0f,
          .resolution = 1.0f,
          .power      = 0.0f,
          .reserved   = {}
        },
#endif
};

static struct sensor_t  sSensorList[MAX_NUM_SENSORS];

static uint32_t sensors__get_sensors_list(struct sensors_module_t* module,
        struct sensor_t const** list) 
{
    int count = 0;
    int nn;
    for (nn = 0; nn < MAX_NUM_SENSORS; nn++) {
        sSensorList[count++] = sSensorListInit[nn];
    }
    *list = sSensorList;
    return MAX_NUM_SENSORS;
}


static int
open_sensors(const struct hw_module_t* module,
             const char*               name,
             struct hw_device_t*      *device)
{
    int  status = -EINVAL;

    D("%s: name=%s", __FUNCTION__, name);

    if (!strcmp(name, SENSORS_HARDWARE_CONTROL))
    {
        SensorControl *dev = malloc(sizeof(*dev));

        memset(dev, 0, sizeof(*dev));

        dev->device.common.tag       = HARDWARE_DEVICE_TAG;
        dev->device.common.version   = 0;
        dev->device.common.module    = (struct hw_module_t*) module;
        dev->device.common.close     = control__close;
        dev->device.open_data_source = control__open_data_source;
        dev->device.activate         = control__activate;
        dev->device.set_delay        = control__set_delay;
        dev->device.wake             = control__wake;
        dev->fd                      = -1;

        *device = &dev->device.common;
        status  = 0;
    }
    else if (!strcmp(name, SENSORS_HARDWARE_DATA)) {
        SensorData *dev = malloc(sizeof(*dev));

        memset(dev, 0, sizeof(*dev));

        dev->device.common.tag     = HARDWARE_DEVICE_TAG;
        dev->device.common.version = 0;
        dev->device.common.module  = (struct hw_module_t*) module;
        dev->device.common.close   = data__close;
        dev->device.data_open      = data__data_open;
        dev->device.data_close     = data__data_close;
        dev->device.poll           = data__poll;
        dev->events_fd             = -1;
        dev->stat                  = 0;
        dev->val                   = 0;
        *device = &dev->device.common;
        status  = 0;
    }
    return status;
}


static struct hw_module_methods_t sensors_module_methods = {
    .open = open_sensors
};

const struct sensors_module_t HAL_MODULE_INFO_SYM = {
    .common = {
        .tag = HARDWARE_MODULE_TAG,
        .version_major = 1,
        .version_minor = 0,
        .id = SENSORS_HARDWARE_MODULE_ID,
        .name = "bc10 SENSORS Module",
        .author = "BeatCraft. inc",
        .methods = &sensors_module_methods,
    },
    .get_sensors_list = sensors__get_sensors_list,
};
